/******************************************************************************
 * Example showing the usage of the C++ library libirimager                   *
 * --------------------------------------------------------                   *
 * With this example up to 4 devices can be instantiated simultaneously.      *
 * For each device, an XML file needs to be passed. A generic XML file having *
 * the serial number == 0, forces the library to search for any valid device. *
 ******************************************************************************/

#include <cstdio>
#include <iostream>
#include <vector>

// IR Imager device interfaces
#include "IRDeviceDS.h"

// IR Imager imager interfaces
#include "IRImager.h"

// IR Imager logging interface
#include "IRLogger.h"

// IR Imager image converter
#include "ImageBuilder.h"

// Raw video interface
#include "VideoCapture.h"

// Visualization
#include "VideoDisplay.h"

// Helper class to measure the achievable frame rate
#include "FramerateCounter.h"

// Helper class to maintain and query calibration data
#include "IRCalibrationManager.h"

evo::IRImager*         _imagers[4];

evo::FramerateCounter* _frc[4];

unsigned char*         _thermalImages[4];

VideoDisplay*          _displays[4];

// Function called be DirectShow interface when a new frame has been acquired.
// Multiple cameras are distinguished by the variable id.
void onRawFrame(unsigned char* data, int len, int id)
{
  double fps;
  if(_frc[id]->trigger(&fps))
    std::cout << "Frame rate: " << fps << " fps" << std::endl;
  int arg = id;
  _imagers[id]->process(data, &arg);
}

// Function called within process call of IRImager instance.
// Keep this function free of heavy processing load. Otherwise the frame rate will drop down significantly for the control loop.
// With the optional argument, one can distinguish between multiple instances.
// A more sophisticated way to do so, is the usage of the object oriented interface (IRImagerClient).
void onThermalFrame(unsigned short* image, unsigned int w, unsigned int h, evo::IRFrameMetadata meta, void* arg)
{
  evo::ImageBuilder iBuilder;
  iBuilder.setPaletteScalingMethod(evo::eMinMax);
  iBuilder.setPalette(evo::eIron);
  iBuilder.setData(w, h, image);
  int* instance = (int*) arg;
  if(_thermalImages[*instance] == NULL)
    _thermalImages[*instance] = new unsigned char[iBuilder.getStride() * h * 3];
  unsigned char* thermalImage = _thermalImages[*instance];

  iBuilder.convertTemperatureToPaletteImage(thermalImage);

  _displays[*instance]->drawCapture(0, 0, iBuilder.getStride(), h, 24, thermalImage);
}

// Function called within process call of IRImager instance, every time the state of the shutter flag changes.
// The flag state changes either automatically or by calling the forceFlagEvent method of IRImager.
void onFlageStateChange(evo::EnumFlagState fs, void* arg)
{
  std::cout << "Flag state for instance " << *((int*)(arg)) << ": " << fs << std::endl;
}

int main(int argc, char* argv[])
{
  std::string filename = "";
  evo::IRLogger::setVerbosity(evo::IRLOG_ERROR, evo::IRLOG_OFF, filename.c_str());

  if(argc < 2 || argc > 5)
  {
    std::cout << "usage: " << argv[0] << " <path to xml file> [path to xml file] [path to xml file] [path to xml file]" << std::endl;
    return -1;
  }

  evo::VideoCapture vc;
  std::vector<evo::IRDeviceDS*> devices;
  std::vector<evo::IRDeviceParams> params;

  for(int i = 1; i <= argc - 1; i++)
  {
    char* args = argv[i];
    // Windows SDK is compiled using 16-bit Unicode characters
    size_t argSize = strlen(args) + 1, converted;
    wchar_t* argPath = new wchar_t[argSize];
    mbstowcs_s(&converted, argPath, argSize, args, argSize);

    evo::IRDeviceParams p;
    if (evo::IRDeviceParamsReader::readXML(argPath, p) == false)
      return -1;
    delete argPath;

    evo::IRDeviceDS* device = vc.initializeDevice(p);

    if(device)
    {
      devices.push_back(device);
      params.push_back(p);

      evo::IRCalibrationManager* caliManager = evo::IRCalibrationManager::getInstance();
      caliManager->setCalibrationDir(p.caliPath);
      caliManager->setFormatsDir(p.formatsPath);

      // Output available optics
      const std::vector<evo::IROptics>* optics = caliManager->getAvailableOptics(device->getSerial());
      std::cout << "Available optics for camera with serial " << device->getSerial() << std::endl;
      for (unsigned int i = 0; i < optics->size(); i++)
      {
        evo::IROptics op = (*optics)[i];
        std::wcout << "FOV: " << op.fov << " deg, Text: " << op.text << std::endl;
        for (unsigned int j = 0; j < op.tempRanges.size(); j++)
        {
          std::cout << " tMin: " << op.tempRanges[j].tMin << " C, tMax: " << op.tempRanges[j].tMax << " C" << std::endl;
        }
      }
      std::cout << std::endl;
    }
  }

  vc.run();

  if (devices.size()>0)
  {
	  for (unsigned int i = 0; i < devices.size(); i++)
    {
      evo::IRDeviceDS* device    = devices[i];
      _imagers[i]                = new evo::IRImager();
      _frc[i]                    = new evo::FramerateCounter(1000.0, device->getFrequency());
      _displays[i]               = NULL;
      _thermalImages[i]          = NULL;

      evo::IRImager* imager      = _imagers[i];
      if(imager->init(&params[i], device->getFrequency(), device->getWidth(), device->getHeight()))
      {
        imager->setFrameCallback(onThermalFrame);
        imager->setFlagStateCallback(onFlageStateChange);
        device->setCallback(onRawFrame);
        device->startStreaming();
        _displays[i] = new VideoDisplay(imager->getWidth(), imager->getHeight());
        if(i<(devices.size()-1))
          _displays[i]->showDetach();
        else
          _displays[i]->show();
      }
    }
  }
  else
  {
    std::cout << "IR Imager device(s) could not be found" << std::endl;
  }

  for(unsigned int i = 0; i < devices.size(); i++)
  {
    devices[i]->stopStreaming();
    delete _frc[i];
    delete _imagers[i];
    if(_thermalImages[i])  delete _thermalImages[i];
    if(_displays[i])       delete _displays[i];
  }

  return 0;
}
